/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System.Windows.Forms;
using Inet.Viewer.Data;
namespace Inet.Viewer
{
    /// <summary>
    /// enum to distinguish the different export formats    
    /// </summary>
    public enum ExportFormat
    {
        /// <summary>
        /// PDF
        /// </summary>
        PDF,
        /// <summary>
        /// Export Format XLS (Microsoft Excel format)
        /// </summary>
        XLS,
        /// <summary>
        /// Open Document Spreadsheet(ODS)
        /// </summary>
        ODS,
        /// <summary>
        /// RTF
        /// </summary>
        RTF,
        /// <summary>
        /// HTML
        /// </summary>
        HTML,
        /// <summary>
        /// Export Format CSV (Comma separated values)
        /// </summary>
        CSV,
        /// <summary>
        /// Export Format DATA (Comma separated values)
        /// </summary>
        DATA,
        /// <summary>
        /// Export Format TXT (Plain Text Format)
        /// </summary>
        TXT,
        /// <summary>
        /// JPG Iamges
        /// </summary>
        JPG,
        /// <summary>
        /// PNG Images
        /// </summary>
        PNG,
        /// <summary>
        /// GIF Images
        /// </summary>
        GIF,
        /// <summary>
        /// PostScript Level 1 
        /// </summary>
        PS,
        /// <summary>
        /// PostScript Level 2 
        /// </summary>
        PS2,
        /// <summary>
        /// PostScript Level 3 
        /// </summary>
        PS3,
        /// <summary>
        /// XML 
        /// </summary>
        XML,
        /// <summary>
        /// Bitmap Images
        /// </summary>
        BMP,
        /// <summary>
        /// Scalable Vector Graphics
        /// </summary>
        SVG,
        /// <summary>
        /// Option for no export format
        /// </summary>
        None
    }

    /// <summary>
    /// This is the top level viewer instance of the i-net Clear Reports report viewer, and allows
    /// creation, retrieval, and removal of one or more ReportViews.
    /// To create a ReportView, a ReportData object is necessary, in order for the ReportView to be
    /// able to fetch its report data.
    /// This class also holds the toolbar which can be used to fire actions, add or remove buttons.
    /// </summary>    
    public interface IReportViewer
    {

        /// <summary>
        /// Returns the currently visible or selected ReportView object. Note that this will
        /// return null if no ReportView is currently visible or selected.
        /// </summary>
        /// <returns> Currently active ReportView, or null if none is currently active</returns>
        IReportView CurrentReportView { get; set; }

        /// <summary>
        /// Returns the number of ReportViews registered and added to this viewer. The return value -1 is the maximum
        /// allowed index for <seealso cref="GetReportView(int)"/>.
        /// </summary>
        /// <returns> The number of ReportViews held by the ReportViewer </returns> 
        int ReportViewCount { get; }

        /// <summary>
        /// Returns the report view at the given index. Note this report view need not be the current report view
        /// or even visible. The maximum allowed index is <code>getReportViewCount()-1</code>, the minimum allowed is 0.
        /// </summary>
        /// <param name="i"> Index of report view to fetch. </param>
        /// <returns> ReportView at the index given </returns>
        /// <exception cref="System.IndexOutOfRangeException"> If index is smaller than 0 or is greater than or equal to getReportViewCount().</exception>
        IReportView GetReportView(int i);


        /// <summary>
        /// Returns the current ToolBar - this tool bar belongs to the currently visible or
        /// selected ReportView object. This object is responsible for various user actions for
        /// navigating through the report, printing, etc.
        /// </summary>
        /// <returns> ToolBar of current ReportView</returns>
        IToolBar ToolBar { get; }

        /// <summary>
        /// Adds a given ReportView to the ReportViewer - this ReportView need not be
        /// initialized at this point in time.<br/>
        /// Note that as long as the viewer exists, this report view will be held in memory, unless it is
        /// removed via <seealso cref="CloseReportView(int)"/> or <seealso cref="CloseAllReportViews"/>. In this case, it can not be added
        /// back to the viewer via this method. Instead it must be recreated with the <seealso cref="IRenderData"/>.<br/>
        /// Note also that the report view will not have a close button if it is the first report view added
        /// to the viewer, otherwise it will. If you'd like to customize this behavior, use the method
        /// <seealso cref="IReportView.IsCloseable"/> instead, where you can manually set whether or not the view
        /// is to have a close button.
        /// </summary>
        /// <param name="view"> ReportView to add </param>
        /// <seealso cref= "CloseReportView(int)"/>
        /// <seealso cref= "CloseReportView(IReportView)"/>
        /// <seealso cref= "CloseAllReportViews()"/>
        /// <seealso cref= "AddReportView(IReportView)"/>    
        /// <seealso cref="CloseReportView(IReportView)"/> or <seealso cref="CloseReportView(int)"/> or <seealso cref="CloseAllReportViews"/>. 
        void AddReportView(IReportView view);

        /// <summary>  
        /// Also adds this newly created ReportView to the ReportViewer.<br/>
        /// Note that as long as the viewer exists, this report view will be held in memory, unless it is
        /// removed via <seealso cref="CloseReportView(int)"/> or <seealso cref="CloseAllReportViews"/>. In this case, it can not be added
        /// back to the Viewer via this method. Instead it must be recreated with the ReportData.<br/>
        /// Note also that the report view will not have a close button if it is the first report view added
        /// to the viewer, otherwise it will. If you'd like to customize this behavior, use the method  
        /// is to have a close button.
        /// <br/>If a report with precisely the same properties has already been added, this will not  create an identical
        /// second view, rather it will give focus to the view already opened. If you want to add two identical reports to the
        /// same viewer, simply add an identifying property to the report views, such as a time stamp. </summary>
        /// <param name="data"> ReportData object specifying the source of report data </param>                
        /// <returns> Newly created ReportView - this ReportView has now been added to a ReportViewer and should not
        /// simply be added somewhere else without first removing it from the viewer. </returns>
        /// <seealso cref= "CloseReportView(int)"/>
        /// <seealso cref= "CloseReportView(IReportView)"/>
        /// <seealso cref= "CloseAllReportViews"/>
        /// <seealso cref= "AddNewReportView(IRenderData)"/>            
        /// <exception cref="System.ArgumentOutOfRangeException"> if this view has previously been removed.</exception>  
        IReportView AddNewReportView(IRenderData data);

        /// <summary>
        /// Permanently closes and removes the ReportView at the given index, 0-based. That is, <code>closeReportView(0)</code> 
        /// will close and remove the first ReportView added to the Viewer, <code>closeReportView(1)</code> the second, etc.
        /// This will also close any currently open connections to the ReportView's ReportData object.
        /// Note that this causes the report view to be disposed of - trying to add it back to the viewer
        /// via <seealso cref="AddReportView(IReportView)"/> will result in an IllegalStateException. Instead
        /// simply create a new report view with the same ReportData.
        /// 
        /// If there is no ReportView at this index, this method will throw a ViewerException.
        /// </summary>
        /// <param name="index"> Index of ReportView to close and remove.</param>
        void CloseReportView(int index);

        /// <summary>
        /// Permanently closes and disposes the ReportView given as the parameter and removes it from the ReportViewer. 
        /// This will also close any currently open connections to the ReportView's ReportData object.
        /// Note that this causes the report view to be disposed of - trying to add it back to the viewer
        /// via <seealso cref="AddReportView(IReportView)"/> will result in an IllegalStateException. Instead
        /// simply create a new report view with the same ReportData.
        /// </summary>
        /// <param name="view"> ReportView to close and remove from the ReportViewer</param>
        void CloseReportView(IReportView view);

        /// <summary>
        /// Closes and removes all ReportViews that are currently added to the ReportViewer.
        /// This will also close any currently open connections to the various ReportData objects to
        /// which the ReportViews are connected.
        /// Note that this causes the report views to be disposed of - trying to add one of them back to the viewer
        /// via <seealso cref="AddReportView(IReportView)"/> will result in an IllegalStateException. Instead
        /// simply create a new report view with the same ReportData.
        /// </summary>
        void CloseAllReportViews();


        /// <summary>
        /// Event Listener for if the  current View changes
        /// </summary>
        event System.EventHandler ViewChanged;

        /// <summary>
        /// Shows or hides to search panel.
        /// </summary>
        bool SearchVisible { set; }

        /// <summary>
        /// Gets the printer settings.
        /// </summary>
        System.Drawing.Printing.PrinterSettings PrinterSettings { get; }
    }
}